"""Seeded random number generator utilities.

The present‑act engine requires deterministic behaviour except when
resolving ties.  This module wraps NumPy's Random Generator with a
simple interface to ensure that all randomness is seeded and reproducible.
"""

from __future__ import annotations

import numpy as np
from typing import Optional


class RNG:
    """Simple wrapper around NumPy's PCG64 RNG.

    Each instance is initialised with a single seed and provides a method
    for drawing random variates.  Additional methods can be added as
    needed to support more complex sampling operations.
    """

    def __init__(self, seed: int) -> None:
        self._seed = int(seed)
        self._rng = np.random.default_rng(self._seed)

    def seed(self, seed: Optional[int] = None) -> None:
        """Reseeds the generator.  If no seed is provided the existing
        seed is reused.  This resets the underlying RNG state.
        """
        if seed is not None:
            self._seed = int(seed)
        self._rng = np.random.default_rng(self._seed)

    def uniform(self, low: float = 0.0, high: float = 1.0, size: int | tuple[int, ...] = 1) -> np.ndarray:
        """Draw samples from a uniform distribution on [low, high)."""
        return self._rng.uniform(low, high, size)

    def choice(self, a, size=None, replace=True, p=None):
        """Return random samples from a 1‑D array-like or int.

        This delegates to ``numpy.random.Generator.choice``.  It is used
        by the simplified simulation to sample meter symbols uniformly.
        """
        return self._rng.choice(a, size=size, replace=replace, p=p)
